/*
 * Decompiled with CFR 0.152.
 */
package voldemort.client.rebalance;

import com.google.common.collect.Lists;
import java.io.File;
import java.io.StringReader;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ExecutorService;
import org.apache.log4j.Logger;
import voldemort.VoldemortException;
import voldemort.client.protocol.admin.AdminClient;
import voldemort.client.protocol.admin.AdminClientConfig;
import voldemort.client.rebalance.OrderedClusterTransition;
import voldemort.client.rebalance.RebalanceClientConfig;
import voldemort.client.rebalance.RebalanceClusterPlan;
import voldemort.client.rebalance.RebalanceNodePlan;
import voldemort.client.rebalance.RebalancePartitionsInfo;
import voldemort.client.rebalance.RebalanceTask;
import voldemort.cluster.Cluster;
import voldemort.cluster.Node;
import voldemort.server.rebalance.VoldemortRebalancingException;
import voldemort.store.StoreDefinition;
import voldemort.utils.RebalanceUtils;
import voldemort.versioning.Versioned;
import voldemort.xml.ClusterMapper;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class RebalanceController {
    private static final Logger logger = Logger.getLogger(RebalanceController.class);
    private static final DecimalFormat decimalFormatter = new DecimalFormat("#.##");
    private final AdminClient adminClient;
    private final RebalanceClientConfig rebalanceConfig;

    public RebalanceController(String bootstrapUrl, RebalanceClientConfig rebalanceConfig) {
        this.adminClient = new AdminClient(bootstrapUrl, (AdminClientConfig)rebalanceConfig);
        this.rebalanceConfig = rebalanceConfig;
    }

    public RebalanceController(Cluster cluster, RebalanceClientConfig config) {
        this.adminClient = new AdminClient(cluster, (AdminClientConfig)config);
        this.rebalanceConfig = config;
    }

    public void rebalance(Cluster targetCluster) {
        Versioned<Cluster> currentVersionedCluster = RebalanceUtils.getLatestCluster(RebalanceUtils.getNodeIds(Lists.newArrayList(this.adminClient.getAdminClientCluster().getNodes())), this.adminClient);
        Cluster currentCluster = currentVersionedCluster.getValue();
        this.rebalance(currentCluster, targetCluster);
    }

    public void rebalance(Cluster currentCluster, Cluster targetCluster) {
        this.adminClient.setAdminClientCluster(targetCluster);
        List<StoreDefinition> storeDefs = RebalanceUtils.getStoreDefinition(targetCluster, this.adminClient);
        this.rebalance(currentCluster, targetCluster, storeDefs);
    }

    public void rebalance(Cluster currentCluster, Cluster targetCluster, List<StoreDefinition> storeDefs) {
        logger.info((Object)("Current cluster : " + currentCluster));
        logger.info((Object)("Final target cluster : " + targetCluster));
        storeDefs = RebalanceUtils.validateRebalanceStore(storeDefs);
        Cluster newCurrentCluster = RebalanceUtils.getClusterWithNewNodes(currentCluster, targetCluster);
        this.adminClient.setAdminClientCluster(newCurrentCluster);
        if (!this.rebalanceConfig.isShowPlanEnabled()) {
            RebalanceUtils.validateClusterState(newCurrentCluster, this.adminClient);
            RebalanceUtils.validateReadOnlyStores(newCurrentCluster, storeDefs, this.adminClient);
            logger.info((Object)("Propagating new cluster " + newCurrentCluster + " to all nodes"));
            RebalanceUtils.propagateCluster(this.adminClient, newCurrentCluster);
        }
        this.rebalancePerClusterTransition(newCurrentCluster, targetCluster, storeDefs);
    }

    private void rebalancePerClusterTransition(Cluster currentCluster, Cluster targetCluster, List<StoreDefinition> storeDefs) {
        HashMap<Node, List<Integer>> stealerToStolenPrimaryPartitions = new HashMap<Node, List<Integer>>();
        int numTasks = 0;
        int numCrossZoneMoves = 0;
        int numPrimaryPartitionMoves = 0;
        ClusterMapper mapper = new ClusterMapper();
        Cluster currentClusterClone = mapper.readCluster(new StringReader(mapper.writeCluster(currentCluster)));
        for (Node stealerNode : targetCluster.getNodes()) {
            List<Integer> stolenPrimaryPartitions = RebalanceUtils.getStolenPrimaryPartitions(currentCluster, targetCluster, stealerNode.getId());
            numPrimaryPartitionMoves += stolenPrimaryPartitions.size();
            stealerToStolenPrimaryPartitions.put(stealerNode, stolenPrimaryPartitions);
            if (stolenPrimaryPartitions.size() <= 0) continue;
            Node stealerNodeUpdated = currentClusterClone.getNodeById(stealerNode.getId());
            for (Integer donatedPrimaryPartition : stolenPrimaryPartitions) {
                Cluster transitionCluster = RebalanceUtils.createUpdatedCluster(currentClusterClone, stealerNodeUpdated, donatedPrimaryPartition);
                stealerNodeUpdated = transitionCluster.getNodeById(stealerNodeUpdated.getId());
                RebalanceClusterPlan rebalanceClusterPlan = new RebalanceClusterPlan(currentClusterClone, transitionCluster, storeDefs, this.rebalanceConfig.isDeleteAfterRebalancingEnabled());
                numCrossZoneMoves += RebalanceUtils.getCrossZoneMoves(transitionCluster, rebalanceClusterPlan);
                numTasks += RebalanceUtils.getTotalMoves(rebalanceClusterPlan);
                currentClusterClone = transitionCluster;
            }
        }
        logger.info((Object)("Total number of primary partition moves - " + numPrimaryPartitionMoves));
        logger.info((Object)("Total number of cross zone moves - " + numCrossZoneMoves));
        logger.info((Object)("Total number of tasks - " + numTasks));
        int tasksCompleted = 0;
        int primaryPartitionId = 0;
        double totalTimeMs = 0.0;
        for (Node stealerNode : targetCluster.getNodes()) {
            List stolenPrimaryPartitions = (List)stealerToStolenPrimaryPartitions.get(stealerNode);
            if (stolenPrimaryPartitions == null || stolenPrimaryPartitions.isEmpty()) {
                RebalanceUtils.printLog(stealerNode.getId(), logger, "No primary partitions to steal");
                continue;
            }
            RebalanceUtils.printLog(stealerNode.getId(), logger, "All primary partitions to steal = " + stolenPrimaryPartitions);
            Node stealerNodeUpdated = currentCluster.getNodeById(stealerNode.getId());
            for (Integer donatedPrimaryPartition : stolenPrimaryPartitions) {
                RebalanceUtils.printLog(stealerNode.getId(), logger, "Working on moving primary partition " + donatedPrimaryPartition);
                Cluster transitionCluster = RebalanceUtils.createUpdatedCluster(currentCluster, stealerNodeUpdated, donatedPrimaryPartition);
                stealerNodeUpdated = transitionCluster.getNodeById(stealerNodeUpdated.getId());
                RebalanceClusterPlan rebalanceClusterPlan = new RebalanceClusterPlan(currentCluster, transitionCluster, storeDefs, this.rebalanceConfig.isDeleteAfterRebalancingEnabled());
                OrderedClusterTransition orderedClusterTransition = new OrderedClusterTransition(currentCluster, transitionCluster, storeDefs, rebalanceClusterPlan);
                RebalanceUtils.printLog(stealerNode.getId(), logger, orderedClusterTransition.toString());
                if (this.rebalanceConfig.hasOutputDirectory()) {
                    RebalanceUtils.dumpCluster(currentCluster, transitionCluster, new File(this.rebalanceConfig.getOutputDirectory()));
                }
                long startTimeMs = System.currentTimeMillis();
                this.rebalancePerPartitionTransition(stealerNode.getId(), orderedClusterTransition);
                currentCluster = transitionCluster;
                double estimatedTimeMs = (totalTimeMs += (double)(System.currentTimeMillis() - startTimeMs)) / (double)(tasksCompleted += RebalanceUtils.getTotalMoves(rebalanceClusterPlan)) * (double)(numTasks - tasksCompleted);
                RebalanceUtils.printLog(stealerNode.getId(), logger, "Completed tasks - " + tasksCompleted + ". Percent done - " + decimalFormatter.format((double)tasksCompleted * 100.0 / (double)numTasks));
                RebalanceUtils.printLog(stealerNode.getId(), logger, "Primary partitions left to move - " + (numPrimaryPartitionMoves - ++primaryPartitionId));
                RebalanceUtils.printLog(stealerNode.getId(), logger, "Estimated time left for completion - " + estimatedTimeMs + " ms ( " + estimatedTimeMs / 3600000.0 + " hours )");
            }
        }
    }

    private void rebalancePerPartitionTransition(int globalStealerNodeId, OrderedClusterTransition orderedClusterTransition) {
        try {
            List<RebalanceNodePlan> rebalanceNodePlanList = orderedClusterTransition.getOrderedRebalanceNodePlanList();
            if (rebalanceNodePlanList.isEmpty()) {
                RebalanceUtils.printLog(globalStealerNodeId, logger, "Skipping rebalance task id " + orderedClusterTransition.getId() + " since it is empty");
                return;
            }
            RebalanceUtils.printLog(globalStealerNodeId, logger, "Starting rebalance task id " + orderedClusterTransition.getId());
            List<RebalancePartitionsInfo> rebalancePartitionPlanList = RebalanceUtils.flattenNodePlans(rebalanceNodePlanList);
            List<StoreDefinition> readOnlyStoreDefs = RebalanceUtils.filterStores(orderedClusterTransition.getStoreDefs(), true);
            List<StoreDefinition> readWriteStoreDefs = RebalanceUtils.filterStores(orderedClusterTransition.getStoreDefs(), false);
            boolean hasReadOnlyStores = readOnlyStoreDefs != null && readOnlyStoreDefs.size() > 0;
            boolean hasReadWriteStores = readWriteStoreDefs != null && readWriteStoreDefs.size() > 0;
            boolean finishedReadOnlyPhase = false;
            List<RebalancePartitionsInfo> filteredRebalancePartitionPlanList = RebalanceUtils.filterPartitionPlanWithStores(rebalancePartitionPlanList, readOnlyStoreDefs);
            this.rebalanceStateChange(globalStealerNodeId, orderedClusterTransition.getCurrentCluster(), orderedClusterTransition.getTargetCluster(), filteredRebalancePartitionPlanList, hasReadOnlyStores, hasReadWriteStores, finishedReadOnlyPhase);
            if (hasReadOnlyStores) {
                this.rebalancePerTaskTransition(globalStealerNodeId, orderedClusterTransition.getCurrentCluster(), filteredRebalancePartitionPlanList, hasReadOnlyStores, hasReadWriteStores, finishedReadOnlyPhase);
            }
            finishedReadOnlyPhase = true;
            filteredRebalancePartitionPlanList = RebalanceUtils.filterPartitionPlanWithStores(rebalancePartitionPlanList, readWriteStoreDefs);
            this.rebalanceStateChange(globalStealerNodeId, orderedClusterTransition.getCurrentCluster(), orderedClusterTransition.getTargetCluster(), filteredRebalancePartitionPlanList, hasReadOnlyStores, hasReadWriteStores, finishedReadOnlyPhase);
            if (hasReadWriteStores) {
                this.rebalancePerTaskTransition(globalStealerNodeId, orderedClusterTransition.getCurrentCluster(), filteredRebalancePartitionPlanList, hasReadOnlyStores, hasReadWriteStores, finishedReadOnlyPhase);
            }
            RebalanceUtils.printLog(globalStealerNodeId, logger, "Successfully terminated rebalance task id " + orderedClusterTransition.getId());
        }
        catch (Exception e) {
            RebalanceUtils.printErrorLog(globalStealerNodeId, logger, "Error in rebalance task id " + orderedClusterTransition.getId() + " - " + e.getMessage(), e);
            throw new VoldemortException("Rebalance failed on rebalance task id " + orderedClusterTransition.getId(), e);
        }
    }

    private void rebalanceStateChange(int globalStealerNodeId, Cluster currentCluster, Cluster transitionCluster, List<RebalancePartitionsInfo> rebalancePartitionPlanList, boolean hasReadOnlyStores, boolean hasReadWriteStores, boolean finishedReadOnlyStores) {
        try {
            if (!hasReadOnlyStores && !hasReadWriteStores) {
                throw new VoldemortException("Cannot get this state since it means there are no stores");
            }
            if (!hasReadOnlyStores && hasReadWriteStores && !finishedReadOnlyStores) {
                RebalanceUtils.printLog(globalStealerNodeId, logger, "Ignoring state change since there are no read-only stores");
            } else if (!hasReadOnlyStores && hasReadWriteStores && finishedReadOnlyStores) {
                RebalanceUtils.printLog(globalStealerNodeId, logger, "Cluster metadata change + rebalance state change");
                if (!this.rebalanceConfig.isShowPlanEnabled()) {
                    this.adminClient.rebalanceStateChange(currentCluster, transitionCluster, rebalancePartitionPlanList, false, true, true, true, true);
                }
            } else if (hasReadOnlyStores && !finishedReadOnlyStores) {
                RebalanceUtils.printLog(globalStealerNodeId, logger, "Rebalance state change");
                if (!this.rebalanceConfig.isShowPlanEnabled()) {
                    this.adminClient.rebalanceStateChange(currentCluster, transitionCluster, rebalancePartitionPlanList, false, false, true, true, true);
                }
            } else if (hasReadOnlyStores && !hasReadWriteStores && finishedReadOnlyStores) {
                RebalanceUtils.printLog(globalStealerNodeId, logger, "Swap + Cluster metadata change");
                if (!this.rebalanceConfig.isShowPlanEnabled()) {
                    this.adminClient.rebalanceStateChange(currentCluster, transitionCluster, rebalancePartitionPlanList, true, true, false, true, true);
                }
            } else {
                RebalanceUtils.printLog(globalStealerNodeId, logger, "Swap + Cluster metadata change + rebalance state change");
                if (!this.rebalanceConfig.isShowPlanEnabled()) {
                    this.adminClient.rebalanceStateChange(currentCluster, transitionCluster, rebalancePartitionPlanList, true, true, true, true, true);
                }
            }
        }
        catch (VoldemortRebalancingException e) {
            RebalanceUtils.printErrorLog(globalStealerNodeId, logger, "Failure while changing rebalancing state", e);
            throw e;
        }
    }

    private void rebalancePerTaskTransition(int globalStealerNodeId, Cluster currentCluster, List<RebalancePartitionsInfo> rebalancePartitionPlanList, boolean hasReadOnlyStores, boolean hasReadWriteStores, boolean finishedReadOnlyStores) {
        RebalanceUtils.printLog(globalStealerNodeId, logger, "Submitting rebalance tasks for " + rebalancePartitionPlanList);
        if (this.rebalanceConfig.isShowPlanEnabled()) {
            return;
        }
        ExecutorService service = RebalanceUtils.createExecutors(this.rebalanceConfig.getMaxParallelRebalancing());
        ArrayList allTasks = Lists.newArrayList();
        ArrayList successfulTasks = Lists.newArrayList();
        ArrayList failedTasks = Lists.newArrayList();
        try {
            this.executeTasks(service, rebalancePartitionPlanList, allTasks);
            RebalanceUtils.printLog(globalStealerNodeId, logger, "All rebalance tasks were submitted (shutting down in " + this.rebalanceConfig.getRebalancingClientTimeoutSeconds() + " sec)");
            RebalanceUtils.executorShutDown(service, this.rebalanceConfig.getRebalancingClientTimeoutSeconds());
            RebalanceUtils.printLog(globalStealerNodeId, logger, "Finished waiting for executors");
            List<Exception> failures = this.filterTasks(allTasks, successfulTasks, failedTasks);
            if (failedTasks.size() > 0) {
                throw new VoldemortRebalancingException("Rebalance task terminated unsuccessfully", failures);
            }
        }
        catch (VoldemortRebalancingException e) {
            logger.error((Object)("Failure while migrating partitions for stealer node " + globalStealerNodeId));
            if (hasReadOnlyStores && hasReadWriteStores && finishedReadOnlyStores) {
                this.adminClient.rebalanceStateChange(null, currentCluster, null, true, true, false, false, false);
            } else if (hasReadWriteStores && finishedReadOnlyStores) {
                this.adminClient.rebalanceStateChange(null, currentCluster, null, false, true, false, false, false);
            }
            throw e;
        }
        finally {
            if (!service.isShutdown()) {
                RebalanceUtils.printErrorLog(globalStealerNodeId, logger, "Could not shutdown service cleanly for node " + globalStealerNodeId, null);
                service.shutdownNow();
            }
        }
    }

    private List<Exception> filterTasks(List<RebalanceTask> allTasks, List<RebalanceTask> successfulTasks, List<RebalanceTask> failedTasks) {
        ArrayList errors = Lists.newArrayList();
        for (RebalanceTask task : allTasks) {
            if (task.hasException()) {
                failedTasks.add(task);
                errors.add(task.getError());
                continue;
            }
            successfulTasks.add(task);
        }
        return errors;
    }

    private void executeTasks(ExecutorService service, List<RebalancePartitionsInfo> rebalancePartitionPlanList, List<RebalanceTask> taskList) {
        for (RebalancePartitionsInfo partitionsInfo : rebalancePartitionPlanList) {
            RebalanceTask rebalanceTask = new RebalanceTask(partitionsInfo, this.rebalanceConfig, this.adminClient);
            taskList.add(rebalanceTask);
            service.execute(rebalanceTask);
        }
    }

    public AdminClient getAdminClient() {
        return this.adminClient;
    }

    public void stop() {
        this.adminClient.stop();
    }
}

