/*
 * Copyright 2009-2011 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package griffon.plugins.slideware

import java.awt.Component
import java.awt.Dimension
import com.lowagie.text.*
import com.lowagie.text.pdf.*
import java.awt.Graphics
import java.awt.image.BufferedImage
import java.awt.Graphics2D
import java.awt.RenderingHints
import java.awt.geom.AffineTransform

/**
 * @author Andres Almiray
 */
class DeckLauncherController extends AbstractDeckController {
    def model

    void mvcGroupInit(Map<String, Object> args) {
        if(!app.config.presentation) {
            app.config.presentation.fileName = Metadata.current.getApplicationName() + '.pdf'
            app.config.presentation.fullScreen = false
            app.config.presentation.screenWidth = '1024'
            app.config.presentation.screenHeight = '768'
        }
    }

    def configAction = { evt = null ->
        withMVCGroup('DeckConfig') { m, v, c ->
            c.show()
        }
    }

    def playAction = { evt = null ->
        execSync { model.busy = true }
        try {
            app.event('LaunchDeckStart')
            def (m, v, c) = createMVCGroup('DeckPlayer')
            app.windowManager.hide('deckLauncherWindow')
            app.event('LaunchDeckEnd')
            c.show()
        } finally {
            execAsync { model.busy = false }
        }
    }

    def printAction = { evt = null ->
        execSync { model.busy = true }
        def (m, v, c) = createMVCGroup('DeckPlayer')
        
        Map settings = [
            fullScreen: app.config.presentation.fullScreen,
            screenWidth: app.config.presentation.screenWidth,
            screenHeight: app.config.presentation.screenHeight
        ]
        
        try {
            // int width = app.config.presentation.printWidth ?: 1024
            // int height = app.config.presentation.printHeight ?: 768
            float ratio = app.config.presentation.printRatio ?: 1.0f
            
            v.deck.layout.skipTransitions = true
            app.config.presentation.fullScreen = false
            // app.config.presentation.screenWidth = width
            // app.config.presentation.screenHeight = height
            c.show()

            Rectangle size = new Rectangle((int)(app.config.presentation.screenWidth * ratio), (int)(app.config.presentation.screenHeight * ratio))
            Document document = new Document(size, 0f, 0f, 0f, 0f)
            PdfWriter.getInstance(document, new FileOutputStream(app.config.presentation.fileName))
            document.open()
            (0..<v.deck.size()).each { i ->
                Slide slide = v.deck[i]
                def imageSet = null
                execSync {imageSet = slide.takeSnapshot() }
                imageSet.each { image ->
                    image = scale(image, ratio)
                    Image img = Image.getInstance(image, null)
                    img.setDpi(600i, 600i)
                    img.setXYRatio(2.5f)
                    document.add(img)
                }
                for(def action: v.slideActions[i]) {
                    def print = true
                    execSync {
                        if(action.maximumNumberOfParameters == 2) {
                            print = action.call(false, true)
                        } else {
                            print = action.call(false)
                        }
                    }
                    if(print == null || print) {
                        imageSet = null
                        execSync {imageSet = slide.takeSnapshot() }
                        imageSet.each { image ->
                            image = scale(image, ratio)
                            Image img = Image.getInstance(image, null)
                            img.setDpi(600i, 600i)
                            img.setXYRatio(2.5f)
                            document.add(img)
                        }
                    }
                }
                execSync {
                    v.deck.layout.next(v.deck)
                }
            }
            document.close()
        } finally {
            app.config.presentation.fullScreen = settings.fullScreen
            //app.config.presentation.screenWidth = settings.screenWidth
            //app.config.presentation.screenHeight = settings.screenHeight
            c.hide()
            execAsync { model.busy = false }
        }
    }

    def quitAction = { evt = null -> 
        app.shutdown()    
    }

    def onDeckPlayerClosed = {
        def groupsToDelete = []
        app.groups.each { name, group ->
            if(name != 'DeckLauncher') groupsToDelete << name
        }
        groupsToDelete.each { name ->
            destroyMVCGroup(name)
        }
    }

    protected java.awt.Image scale(java.awt.Image image, float ratio) {
        int width = image.getWidth(null) * ratio;
        int height = image.getHeight(null) * ratio;
        BufferedImage img = GraphicsUtil.createCompatibleImage(width, height, false)
        Graphics2D g = (Graphics2D)img.getGraphics();
        g.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_QUALITY)
        g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON)
        g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON)
        g.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC)
        g.drawImage(image, AffineTransform.getScaleInstance(ratio, ratio), null)
        return img
    }
}
